package main

import (
	"math/rand"

	"github.com/veandco/go-sdl2/sdl"
)

const (
	jetBeamParticleEffectParticleWidth     int32   = 8
	jetBeamParticleEffectParticleHeight    int32   = 8
	jetBeamParticleEffectParticleVelocityX float32 = 0.0
	jetBeamParticleEffectParticleVelocityY float32 = 0.2

	jetBeamParticleEffectYellowParticleMinAliveTime float32 = 50
	jetBeamParticleEffectYellowParticleMaxAliveTime float32 = 100
	jetBeamParticleEffectYellowParticleRed          uint8   = 255
	jetBeamParticleEffectYellowParticleGreen        uint8   = 255
	jetBeamParticleEffectYellowParticleBlue         uint8   = 115

	jetBeamParticleEffectOrangeParticleMinAliveTime float32 = 100
	jetBeamParticleEffectOrangeParticleMaxAliveTime float32 = 150
	jetBeamParticleEffectOrangeParticleRed          uint8   = 255
	jetBeamParticleEffectOrangeParticleGreen        uint8   = 160
	jetBeamParticleEffectOrangeParticleBlue         uint8   = 40
)

type JetBeamParticleEffect struct {
	originX         float32
	originY         float32
	effectWidth     int
	effectHeight    int
	yellowParticles []*Particle
	orangeParticles []*Particle
}

func NewJetBeamParticleEffect(x float32, y float32, w, h, yellow, orange int) *JetBeamParticleEffect {
	jetBeam := &JetBeamParticleEffect{}
	jetBeam.originX = x
	jetBeam.originY = y
	jetBeam.effectWidth = w
	jetBeam.effectHeight = h

	for i := 1; i <= yellow; i++ {
		particle := jetBeam.newYellowParticle(jetBeam.originX, jetBeam.originY)
		jetBeam.yellowParticles = append(jetBeam.yellowParticles, particle)
	}
	for i := 1; i <= orange; i++ {
		particle := jetBeam.newOrangeParticle(jetBeam.originX, jetBeam.originY)
		jetBeam.orangeParticles = append(jetBeam.orangeParticles, particle)
	}

	return jetBeam
}

func (jetBeam *JetBeamParticleEffect) newYellowParticle(x float32, y float32) *Particle {
	return jetBeam.newParticle(x, y,
		jetBeamParticleEffectYellowParticleMinAliveTime,
		jetBeamParticleEffectYellowParticleMaxAliveTime,
		jetBeamParticleEffectYellowParticleRed,
		jetBeamParticleEffectYellowParticleGreen,
		jetBeamParticleEffectYellowParticleBlue)
}

func (jetBeam *JetBeamParticleEffect) newOrangeParticle(x float32, y float32) *Particle {
	return jetBeam.newParticle(x, y,
		jetBeamParticleEffectOrangeParticleMinAliveTime,
		jetBeamParticleEffectOrangeParticleMaxAliveTime,
		jetBeamParticleEffectOrangeParticleRed,
		jetBeamParticleEffectOrangeParticleGreen,
		jetBeamParticleEffectOrangeParticleBlue)
}

func (jetBeam *JetBeamParticleEffect) newParticle(x float32, y float32,
	minAlive float32, maxAlive float32, red uint8, green uint8, blue uint8) *Particle {

	particleX := float32(rand.Intn(jetBeam.effectWidth)) + x
	particleY := float32(rand.Intn(jetBeam.effectHeight)) + y
	aliveTime := float32(rand.Intn(int(maxAlive-minAlive))) + minAlive

	particle := &Particle{}
	particle.rectangle.X = 0
	particle.rectangle.Y = 0
	particle.rectangle.W = jetBeamParticleEffectParticleWidth
	particle.rectangle.H = jetBeamParticleEffectParticleHeight
	particle.alive = true
	particle.aliveTime = aliveTime
	particle.timeLeft = aliveTime
	particle.red = red
	particle.green = green
	particle.blue = blue
	particle.originX = particleX
	particle.originY = particleY
	particle.currentX = particleX
	particle.currentY = particleY
	particle.velocityX = jetBeamParticleEffectParticleVelocityX
	particle.velocityY = jetBeamParticleEffectParticleVelocityY

	return particle
}

func (jetBeam *JetBeamParticleEffect) updateParticle(particle *Particle, deltaTime float32, x, y float32) {
	particle.Update(deltaTime)
	if !particle.IsAlive() {
		particle.originX = float32(rand.Intn(jetBeam.effectWidth)) + x
		particle.originY = float32(rand.Intn(jetBeam.effectHeight)) + y
		particle.Reset()
	}
}

func (jetBeam *JetBeamParticleEffect) Update(deltaTime float32, x, y float32) {
	for _, particle := range jetBeam.yellowParticles {
		jetBeam.updateParticle(particle, deltaTime, x, y)
	}
	for _, particle := range jetBeam.orangeParticles {
		jetBeam.updateParticle(particle, deltaTime, x, y)
	}
}

func (jetBeam *JetBeamParticleEffect) Draw(renderer *sdl.Renderer) {
	for _, particle := range jetBeam.orangeParticles {
		particle.Draw(renderer)
	}
	for _, particle := range jetBeam.yellowParticles {
		particle.Draw(renderer)
	}
}
